const express = require('express');
const router = express.Router();
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const multer = require('multer');
const path = require('path');
const fs = require('fs-extra');
const { getPool, initDBWithConfig } = require('./db');
const logger = require('./logger');
const { 
  apiLimiter, 
  adminLimiter,
  authLimiter, 
  uploadLimiter, 
  sanitizeBody, 
  sanitizeQuery,
  validateLogin,
  validateMessage,
  validateProject,
  validateBlogPost
} = require('./middleware');

const SECRET_KEY = process.env.JWT_SECRET || 'viraving-secret-key-change-in-production';

// Apply rate limiting to all API routes (except settings and install which use adminLimiter)
router.use((req, res, next) => {
  // Skip rate limiting for settings and install endpoints (they use adminLimiter)
  if (req.path === '/settings' || req.path.startsWith('/install')) {
    return next();
  }
  // Apply general rate limiting to all other routes
  apiLimiter(req, res, next);
});

// Helper to safely get pool or return null
const safeGetPool = () => {
  try {
    return getPool();
  } catch (error) {
    return null;
  }
};

// --- MIDDLEWARE ---
const authenticateToken = (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];

  if (!token) return res.sendStatus(401);

  jwt.verify(token, SECRET_KEY, (err, user) => {
    if (err) return res.sendStatus(403);
    req.user = user;
    next();
  });
};

// --- FILE UPLOAD CONFIG ---
// In Liara production, the application filesystem (/app) is read-only.
// We use a writable disk mount (e.g. /storage/uploads) configured via Liara.
const uploadsRoot =
  process.env.UPLOADS_DIR ||
  (process.env.NODE_ENV === 'production'
    ? '/storage/uploads'
    : path.join(__dirname, 'public/uploads'));

const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, uploadsRoot);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, uniqueSuffix + path.extname(file.originalname));
  }
});
const upload = multer({ storage: storage });

// --- HELPERS ---

const safeParseJson = (value, fallback) => {
  if (!value) return fallback;
  if (typeof value === 'object') return value;
  try {
    return JSON.parse(value);
  } catch {
    return fallback;
  }
};

// --- HEALTH CHECK ---
router.get('/health', async (req, res) => {
  try {
    const pool = safeGetPool();
    let dbStatus = 'disconnected';
    
    if (pool) {
      try {
        await pool.execute('SELECT 1');
        dbStatus = 'connected';
      } catch (err) {
        dbStatus = 'error';
      }
    }
    
    res.json({
      status: 'ok',
      timestamp: new Date().toISOString(),
      database: dbStatus,
      uptime: process.uptime(),
      memory: {
        used: Math.round(process.memoryUsage().heapUsed / 1024 / 1024),
        total: Math.round(process.memoryUsage().heapTotal / 1024 / 1024)
      }
    });
  } catch (error) {
    logger.error('Health check error', { error: error.message });
    res.status(500).json({ status: 'error', message: error.message });
  }
});

// --- AUTH ROUTES ---

router.post('/auth/login', authLimiter, sanitizeBody, validateLogin, async (req, res) => {
  try {
    const { username, password } = req.body;
    const pool = safeGetPool();
    if (!pool) {
      return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
    }
    
    const [users] = await pool.execute(
      'SELECT * FROM users WHERE username = ?',
      [username]
    );

    if (users.length === 0) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }

    const user = users[0];
    const isValid = await bcrypt.compare(password, user.password);

    if (isValid) {
      const token = jwt.sign({ username: user.username, id: user.id }, SECRET_KEY, { expiresIn: '24h' });
      logger.info('User logged in successfully', { username: user.username });
      res.json({ token });
    } else {
      logger.warn('Failed login attempt', { username });
      res.status(401).json({ message: 'Invalid credentials' });
    }
  } catch (error) {
    logger.error('Login error', { error: error.message, stack: error.stack });
    res.status(500).json({ message: 'Server error' });
  }
});

router.put('/auth/profile', authenticateToken, async (req, res) => {
  try {
    const { username, password, email } = req.body;
    const pool = safeGetPool();
    if (!pool) {
      return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
    }
    const currentUsername = req.user.username;

    let updateFields = [];
    let updateValues = [];

    if (username && username !== currentUsername) {
      updateFields.push('username = ?');
      updateValues.push(username);
    }

    if (password) {
      const hashedPassword = await bcrypt.hash(password, 10);
      updateFields.push('password = ?');
      updateValues.push(hashedPassword);
    }

    if (email) {
      updateFields.push('email = ?');
      updateValues.push(email);
    }

    if (updateFields.length === 0) {
      return res.json({ success: true });
    }

    updateValues.push(currentUsername);

    await pool.execute(
      `UPDATE users SET ${updateFields.join(', ')} WHERE username = ?`,
      updateValues
    );

    logger.info('Profile updated', { username: req.user.username });
    res.json({ success: true });
  } catch (error) {
    logger.error('Profile update error', { error: error.message, username: req.user.username });
    res.status(500).json({ message: 'Server error' });
  }
});

// --- UPLOAD ROUTE ---
router.post('/upload', authenticateToken, uploadLimiter, upload.single('file'), (req, res) => {
  if (!req.file) {
    logger.warn('Upload attempt without file', { username: req.user?.username });
    return res.status(400).send('No file uploaded.');
  }
  const fileUrl = `${req.protocol}://${req.get('host')}/uploads/${req.file.filename}`;
  logger.info('File uploaded successfully', { filename: req.file.filename, username: req.user?.username });
  res.json({ url: fileUrl, filename: req.file.filename });
});

// --- MEDIA LIBRARY ROUTES ---

// List uploaded media files
router.get('/media', authenticateToken, sanitizeQuery, async (req, res) => {
  try {
    const uploadsDir = uploadsRoot;
    await fs.ensureDir(uploadsDir);
    const files = await fs.readdir(uploadsDir);

    const imageExtensions = ['.jpg', '.jpeg', '.png', '.webp', '.gif', '.avif'];

    const media = await Promise.all(
      files
        .filter((file) => imageExtensions.includes(path.extname(file).toLowerCase()))
        .map(async (file) => {
          const filePath = path.join(uploadsDir, file);
          const stat = await fs.stat(filePath);
          return {
            filename: file,
            url: `${req.protocol}://${req.get('host')}/uploads/${file}`,
            size: stat.size,
            createdAt: stat.birthtime || stat.ctime
          };
        })
    );

    // Sort by newest first
    media.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

    res.json(media);
  } catch (error) {
    logger.error('Error listing media files', { error: error.message, stack: error.stack });
    res.status(500).json({ message: 'Server error' });
  }
});

// Delete a media file
router.delete('/media/:filename', authenticateToken, sanitizeQuery, async (req, res) => {
  try {
    const filename = req.params.filename;
    if (!filename) {
      return res.status(400).json({ message: 'Filename is required' });
    }

    const uploadsDir = uploadsRoot;
    const filePath = path.join(uploadsDir, filename);

    const exists = await fs.pathExists(filePath);
    if (!exists) {
      return res.status(404).json({ message: 'File not found' });
    }

    await fs.remove(filePath);
    res.json({ success: true });
  } catch (error) {
    logger.error('Error deleting media file', { error: error.message, stack: error.stack });
    res.status(500).json({ message: 'Server error' });
  }
});

// --- GENERIC CRUD HELPER ---
const createCrudRoutes = (resourceName, tableName) => {
  // GET ALL (With Pagination & Search)
  router.get(`/${resourceName}`, sanitizeQuery, async (req, res) => {
    try {
      const pool = safeGetPool();
      if (!pool) {
        return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
      }
      const { search, category, page = 1, limit = 100 } = req.query;
      
      let query = `SELECT * FROM ${tableName}`;
      let conditions = [];
      let params = [];

      if (search) {
        if (tableName === 'projects' || tableName === 'blog') {
          conditions.push('(title LIKE ? OR description LIKE ?)');
          const searchTerm = `%${search}%`;
          params.push(searchTerm, searchTerm);
        } else if (tableName === 'messages') {
          conditions.push('(name LIKE ? OR description LIKE ?)');
          const searchTerm = `%${search}%`;
          params.push(searchTerm, searchTerm);
        } else if (tableName === 'team' || tableName === 'reviews') {
          conditions.push('name LIKE ?');
          params.push(`%${search}%`);
        }
      }

      if (category && category !== 'All' && tableName === 'projects') {
        conditions.push('category = ?');
        params.push(category);
      }

      if (conditions.length > 0) {
        query += ' WHERE ' + conditions.join(' AND ');
      }

      // Get total count
      const countQuery = query.replace('SELECT *', 'SELECT COUNT(*) as total');
      const [countResult] = await pool.execute(countQuery, params);
      const total = countResult[0].total;

      // Add pagination
      const offset = (page - 1) * limit;
      query += ` ORDER BY created_at DESC LIMIT ? OFFSET ?`;
      params.push(parseInt(limit), offset);

      const [items] = await pool.execute(query, params);

      // Parse JSON fields safely (handles both plain text and JSON)
      const parsedItems = items.map(item => {
        const parsed = { ...item };
        parsed.tags = safeParseJson(parsed.tags, []);
        parsed.seo = safeParseJson(parsed.seo, {});
        parsed.challenges = safeParseJson(parsed.challenges, []);
        parsed.services = safeParseJson(parsed.services, []);
        parsed.socials = safeParseJson(parsed.socials, {});
        return parsed;
      });

      res.json({ data: parsedItems, total, page: Number(page), limit: Number(limit) });
    } catch (error) {
      logger.error(`Error fetching ${resourceName}`, { error: error.message, stack: error.stack, resourceName });
      res.status(500).json({ message: 'Server error' });
    }
  });

  // GET ONE
  router.get(`/${resourceName}/:id`, sanitizeQuery, async (req, res) => {
    try {
      const pool = safeGetPool();
      if (!pool) {
        return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
      }
      const [items] = await pool.execute(
        `SELECT * FROM ${tableName} WHERE id = ?`,
        [req.params.id]
      );

      if (items.length === 0) {
        return res.status(404).send('Not found');
      }

      const item = items[0];
      // Parse JSON fields safely (handles both plain text and JSON)
      item.tags = safeParseJson(item.tags, []);
      item.seo = safeParseJson(item.seo, {});
      item.challenges = safeParseJson(item.challenges, []);
      item.services = safeParseJson(item.services, []);
      item.socials = safeParseJson(item.socials, {});

      res.json(item);
    } catch (error) {
      logger.error(`Error fetching ${resourceName}`, { error: error.message, stack: error.stack, resourceName });
      res.status(500).json({ message: 'Server error' });
    }
  });

  // CREATE
  router.post(`/${resourceName}`, async (req, res) => {
    try {
      const pool = safeGetPool();
      if (!pool) {
        return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
      }
      const item = req.body;
      const id = item.id || Date.now().toString();

      // Prepare data for insertion
      const fields = ['id'];
      const values = [id];
      const placeholders = ['?'];

      // Map fields based on table
      const fieldMap = {
        projects: ['title', 'category', 'image', 'description', 'tags', 'client', 'year', 'projectUrl', 'seo', 'challenges'],
        blog: ['title', 'excerpt', 'content', 'date', 'readTime', 'image', 'author', 'seo'],
        services: ['title', 'description', 'iconName'],
        process: ['title', 'description', 'iconName'],
        team: ['name', 'role', 'image', 'bio', 'socials'],
        reviews: ['name', 'role', 'text', 'avatar'],
        messages: ['name', 'phone', 'services', 'description', 'date', 'isRead']
      };

      const tableFields = fieldMap[tableName] || [];
      
      for (const field of tableFields) {
        if (item[field] !== undefined) {
          fields.push(field);
          if (typeof item[field] === 'object') {
            values.push(JSON.stringify(item[field]));
          } else {
            values.push(item[field]);
          }
          placeholders.push('?');
        }
      }

      // Special handling for messages
      if (tableName === 'messages' && !item.date) {
        fields.push('date');
        values.push(new Date().toLocaleDateString('fa-IR'));
        placeholders.push('?');
      }

      const query = `INSERT INTO ${tableName} (${fields.join(', ')}) VALUES (${placeholders.join(', ')})`;
      await pool.execute(query, values);

      res.json({ ...item, id });
    } catch (error) {
      logger.error(`Error creating ${resourceName}`, { error: error.message, stack: error.stack, resourceName });
      res.status(500).json({ message: 'Server error' });
    }
  });

  // UPDATE
  router.put(`/${resourceName}`, authenticateToken, sanitizeBody, async (req, res) => {
    try {
      const pool = safeGetPool();
      if (!pool) {
        return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
      }
      const item = req.body;

      if (!item.id) {
        return res.status(400).json({ message: 'ID is required' });
      }

      // Build update query
      const updates = [];
      const values = [];

      const fieldMap = {
        projects: ['title', 'category', 'image', 'description', 'tags', 'client', 'year', 'projectUrl', 'seo', 'challenges'],
        blog: ['title', 'excerpt', 'content', 'date', 'readTime', 'image', 'author', 'seo'],
        services: ['title', 'description', 'iconName'],
        process: ['title', 'description', 'iconName'],
        team: ['name', 'role', 'image', 'bio', 'socials'],
        reviews: ['name', 'role', 'text', 'avatar'],
        messages: ['name', 'phone', 'services', 'description', 'date', 'isRead']
      };

      const tableFields = fieldMap[tableName] || [];

      for (const field of tableFields) {
        if (item[field] !== undefined) {
          updates.push(`${field} = ?`);
          if (typeof item[field] === 'object') {
            values.push(JSON.stringify(item[field]));
          } else {
            values.push(item[field]);
          }
        }
      }

      if (updates.length === 0) {
        return res.status(400).json({ message: 'No fields to update' });
      }

      values.push(item.id);

      await pool.execute(
        `UPDATE ${tableName} SET ${updates.join(', ')} WHERE id = ?`,
        values
      );

      // Fetch updated item
      const [updated] = await pool.execute(
        `SELECT * FROM ${tableName} WHERE id = ?`,
        [item.id]
      );

      if (updated.length === 0) {
        return res.status(404).send('Not found');
      }

      const result = updated[0];
      // Parse JSON fields
      if (result.tags) result.tags = safeParseJson(result.tags, []);
      if (result.seo) result.seo = safeParseJson(result.seo, {});
      if (result.challenges) result.challenges = safeParseJson(result.challenges, []);
      if (result.services) result.services = safeParseJson(result.services, []);
      if (result.socials) result.socials = safeParseJson(result.socials, {});

      res.json(result);
    } catch (error) {
      logger.error(`Error updating ${resourceName}`, { error: error.message, stack: error.stack, resourceName });
      res.status(500).json({ message: 'Server error' });
    }
  });

  // DELETE
  router.delete(`/${resourceName}/:id`, authenticateToken, async (req, res) => {
    try {
      const pool = safeGetPool();
      if (!pool) {
        return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
      }
      const [result] = await pool.execute(
        `DELETE FROM ${tableName} WHERE id = ?`,
        [req.params.id]
      );

      if (result.affectedRows === 0) {
        return res.status(404).send('Not found');
      }

      res.json({ success: true });
    } catch (error) {
      logger.error(`Error deleting ${resourceName}`, { error: error.message, stack: error.stack, resourceName });
      res.status(500).json({ message: 'Server error' });
    }
  });
};

// --- REGISTER CRUD ROUTES ---
createCrudRoutes('projects', 'projects');
createCrudRoutes('blog', 'blog');
createCrudRoutes('services', 'services');
createCrudRoutes('process', 'process');
createCrudRoutes('team', 'team');
createCrudRoutes('reviews', 'reviews');
createCrudRoutes('messages', 'messages');

// --- SPECIFIC ROUTES ---

// Mark Message Read
router.put('/messages/:id/read', authenticateToken, async (req, res) => {
  try {
    const pool = safeGetPool();
    if (!pool) {
      return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
    }
    await pool.execute(
      'UPDATE messages SET isRead = TRUE WHERE id = ?',
      [req.params.id]
    );
    res.json({ success: true });
  } catch (error) {
    logger.error('Error marking message as read', { error: error.message, stack: error.stack });
    res.status(500).json({ message: 'Server error' });
  }
});

// Default settings
const getDefaultSettings = () => ({
  siteName: 'ویراوینگ',
  seo: { title: 'ویراوینگ', description: 'طراحی سایت و اپلیکیشن', keywords: ['web', 'app'] },
  hero: { badge: 'آینده وب اینجاست', title: 'ما رویاهای دیجیتال شما را', highlightedWord: 'خلق می‌کنیم', description: 'توسعه تخصصی وب و موبایل', ctaText: 'مشاوره رایگان' },
  footer: { socials: {}, contact: { address: 'تهران', phone: '021-88881234', email: 'info@viraving.com' } },
  navigation: { 
    header: [
      { id: '1', name: 'خانه', href: '#home' },
      { id: '2', name: 'پروژه‌ها', href: '#portfolio' },
      { id: '3', name: 'وبلاگ', href: '#blog' },
      { id: '4', name: 'مشاوره هوشمند', href: '#home', anchor: 'ai-consultant' },
      { id: '5', name: 'تماس با ما', href: '#contact', special: true },
    ],
    footer: {
      quickAccess: [
        { id: 'f1', name: 'خانه', href: '#home' },
        { id: 'f2', name: 'نمونه کارها', href: '#portfolio' },
        { id: 'f3', name: 'مشاوره هوشمند', href: '#home', anchor: 'ai-consultant' },
        { id: 'f4', name: 'شروع همکاری', href: '#contact' },
      ],
      services: [
        { id: 's1', name: 'طراحی وب‌سایت', href: '#home', anchor: 'services' },
        { id: 's2', name: 'توسعه اپلیکیشن', href: '#home', anchor: 'services' },
        { id: 's3', name: 'طراحی UI/UX', href: '#home', anchor: 'services' },
        { id: 's4', name: 'سئو و مارکتینگ', href: '#home', anchor: 'services' },
      ]
    },
    mobile: [
      { id: 'm1', name: 'خانه', href: '#home', iconName: 'Home' },
      { id: 'm2', name: 'پروژه‌ها', href: '#portfolio', iconName: 'Briefcase' },
      { id: 'm3', name: 'مشاور AI', href: '#home', anchor: 'ai-consultant', iconName: 'Zap' },
      { id: 'm4', name: 'تماس', href: '#contact', iconName: 'Phone' },
    ]
  },
  aiConfig: { enabled: true, provider: 'liara', modelName: 'openai/gpt-5-mini:online', systemInstruction: '', apiKey: '' },
  legal: { terms: '', privacy: '' }
});

// Settings
router.get('/settings', adminLimiter, async (req, res) => {
  try {
    const pool = safeGetPool();
    if (!pool) {
      // Return default settings if DB not initialized
      logger.warn('Database pool not available, returning default settings');
      return res.json(getDefaultSettings());
    }
    
    const [rows] = await pool.execute('SELECT data FROM settings ORDER BY id DESC LIMIT 1');
    
    if (rows.length === 0) {
      // Return default settings
      logger.warn('No settings found in database, returning default settings');
      return res.json(getDefaultSettings());
    }

    // Handle both JSON string and object
    let settings;
    try {
      if (typeof rows[0].data === 'string') {
        settings = JSON.parse(rows[0].data);
      } else {
        // Already an object (MySQL JSON type)
        settings = rows[0].data;
      }
    } catch (parseError) {
      logger.error('Error parsing settings, returning default', { error: parseError.message });
      return res.json(getDefaultSettings());
    }
    
    res.json(settings);
  } catch (error) {
    logger.error('Error fetching settings', { error: error.message, stack: error.stack });
    // Return default settings on error instead of 500
    res.json(getDefaultSettings());
  }
});

router.put('/settings', adminLimiter, authenticateToken, sanitizeBody, async (req, res) => {
  try {
    const pool = safeGetPool();
    if (!pool) {
      return res.status(503).json({ message: 'Database not initialized. Please complete installation.' });
    }
    const settingsData = JSON.stringify(req.body);
    
    // Check if settings exist
    const [existing] = await pool.execute('SELECT id FROM settings ORDER BY id DESC LIMIT 1');
    
    if (existing.length > 0) {
      await pool.execute('UPDATE settings SET data = ? WHERE id = ?', [settingsData, existing[0].id]);
    } else {
      await pool.execute('INSERT INTO settings (data) VALUES (?)', [settingsData]);
    }

    res.json(req.body);
  } catch (error) {
    logger.error('Error updating settings', { error: error.message, stack: error.stack });
    res.status(500).json({ message: 'Server error' });
  }
});

// AI Consultant Route (Liara AI)
router.post('/ai/generate-plan', sanitizeBody, async (req, res) => {
  try {
    const { idea, customInstruction } = req.body;
    
    if (!idea || !idea.trim()) {
      return res.status(400).json({ message: 'ایده پروژه الزامی است' });
    }

    const pool = safeGetPool();
    if (!pool) {
      return res.status(503).json({ message: 'Database not initialized' });
    }

    // Get settings from database
    const [rows] = await pool.execute('SELECT data FROM settings ORDER BY id DESC LIMIT 1');
    
    if (rows.length === 0) {
      return res.status(500).json({ message: 'Settings not found' });
    }

    let settings;
    try {
      settings = typeof rows[0].data === 'string' 
        ? JSON.parse(rows[0].data)
        : rows[0].data;
    } catch (parseError) {
      return res.status(500).json({ message: 'Error parsing settings' });
    }

    const apiKey = settings?.aiConfig?.apiKey || '';
    
    if (!apiKey) {
      return res.status(400).json({ 
        message: 'کلید API هوش مصنوعی تنظیم نشده است. لطفاً در بخش تنظیمات پنل ادمین، کلید API Liara را وارد کنید.' 
      });
    }

    if (!settings?.aiConfig?.enabled) {
      return res.status(400).json({ message: 'مشاوره هوشمند غیرفعال است' });
    }

    // Liara AI Configuration
    let OpenAI;
    try {
      OpenAI = require('openai');
    } catch (requireError) {
      logger.error('OpenAI package not found', { error: requireError.message });
      return res.status(500).json({ 
        message: 'پکیج OpenAI نصب نشده است. لطفاً در backend اجرا کنید: npm install openai' 
      });
    }
    
    const baseURL = 'https://ai.liara.ir/api/69492c43a99a2016cacea976/v1';
    // Get model name and ensure it's not HTML-escaped
    let modelName = settings?.aiConfig?.modelName || 'openai/gpt-5-mini:online';
    // Decode HTML entities if present (fix for previously escaped model names)
    if (modelName.includes('&#x2F;') || modelName.includes('&#47;')) {
      modelName = modelName.replace(/&#x2F;/g, '/').replace(/&#47;/g, '/');
    }

    const openai = new OpenAI({
      baseURL: baseURL,
      apiKey: apiKey,
    });

    const defaultInstruction = `
      شما یک مشاور فنی ارشد در شرکت نرم‌افزاری "ویراوینگ" هستید.
      وظیفه شما تحلیل ایده پروژه کاربر و پیشنهاد یک استک تکنولوژی مدرن و یک نقشه راه (Roadmap) کلی است.
      پاسخ باید کاملا به زبان فارسی باشد، اما کلمات فنی (مثل React, Node.js) به انگلیسی نوشته شوند.
      لحن باید حرفه‌ای، دلگرم‌کننده و تخصصی باشد.
    `;

    const systemInstruction = customInstruction || settings?.aiConfig?.systemInstruction || defaultInstruction;

    const prompt = `
      ایده پروژه کاربر: "${idea}"
      
      لطفا موارد زیر را مشخص کن:
      1. استک فنی پیشنهادی (فرانت‌اند، بک‌اند، دیتابیس، و ابزارهای جانبی).
      2. نقشه راه در 3 فاز (MVP، توسعه، مقیاس‌دهی).
      3. یک خلاصه مدیریتی کوتاه درباره پتانسیل این پروژه.
      
      پاسخ را به صورت JSON با ساختار زیر برگردان:
      {
        "techStack": {
          "frontend": "...",
          "backend": "...",
          "database": "...",
          "extras": ["...", "..."]
        },
        "roadmap": [
          {
            "phase": "...",
            "duration": "...",
            "description": "..."
          }
        ],
        "summary": "..."
      }
    `;

    let completion;
    try {
      completion = await openai.chat.completions.create({
        model: modelName,
        messages: [
          {
            role: 'system',
            content: systemInstruction,
          },
          {
            role: 'user',
            content: prompt,
          },
        ],
        response_format: { type: "json_object" },
        temperature: 0.7,
      });
    } catch (openaiError) {
      logger.error('OpenAI API call error', { 
        error: openaiError.message, 
        stack: openaiError.stack,
        model: modelName,
        baseURL: baseURL
      });
      return res.status(500).json({ 
        message: `خطا در ارتباط با Liara AI: ${openaiError.message || 'خطای نامشخص'}` 
      });
    }

    const content = completion.choices[0]?.message?.content;
    if (!content) {
      logger.error('Empty response from AI', { completion });
      return res.status(500).json({ message: 'پاسخ خالی از AI دریافت شد' });
    }

    let parsed;
    try {
      parsed = JSON.parse(content);
    } catch (parseError) {
      logger.error('JSON parse error', { 
        error: parseError.message, 
        content: content.substring(0, 200) 
      });
      return res.status(500).json({ 
        message: 'خطا در پردازش پاسخ AI. پاسخ معتبر JSON نیست.' 
      });
    }

    // Validate response structure
    if (!parsed.techStack || !parsed.roadmap || !parsed.summary) {
      logger.warn('Invalid AI response structure', { parsed });
      // Try to fix or return error
      return res.status(500).json({ 
        message: 'ساختار پاسخ AI معتبر نیست. لطفاً دوباره تلاش کنید.' 
      });
    }

    logger.info('AI plan generated successfully', { idea: idea.substring(0, 50) });
    res.json(parsed);

  } catch (error) {
    logger.error('AI plan generation error', { 
      error: error.message, 
      stack: error.stack,
      name: error.name 
    });
    res.status(500).json({ 
      message: error.message || 'خطا در ارتباط با هوش مصنوعی.' 
    });
  }
});

// Dashboard Stats
router.get('/stats', async (req, res) => {
  try {
    const pool = safeGetPool();
    if (!pool) {
      return res.json({ projects: 0, blog: 0, messages: 0 });
    }
    const [projects] = await pool.execute('SELECT COUNT(*) as count FROM projects');
    const [blog] = await pool.execute('SELECT COUNT(*) as count FROM blog');
    const [messages] = await pool.execute('SELECT COUNT(*) as count FROM messages');

    res.json({
      projects: projects[0].count,
      blog: blog[0].count,
      messages: messages[0].count
    });
  } catch (error) {
    logger.error('Error fetching stats', { error: error.message, stack: error.stack });
    res.status(500).json({ message: 'Server error' });
  }
});

// Check Installation Status
router.get('/install/status', adminLimiter, async (req, res) => {
  try {
    const pool = safeGetPool();
    if (!pool) {
      return res.json({ installed: false });
    }
    
    // Check if users table exists and has at least one user
    const [users] = await pool.execute('SELECT COUNT(*) as count FROM users');
    const [settings] = await pool.execute('SELECT COUNT(*) as count FROM settings');
    
    const isInstalled = users[0].count > 0 && settings[0].count > 0;
    res.json({ installed: isInstalled });
  } catch (error) {
    // If error (e.g., tables don't exist), app is not installed
    logger.error('Install status check error', { error: error.message, stack: error.stack });
    res.json({ installed: false });
  }
});

// Install Route (Initial Setup)
router.post('/install', adminLimiter, async (req, res) => {
  try {
    const { dbConfig, adminUser } = req.body;

    // Initialize database with provided config
    await initDBWithConfig(dbConfig);

    // Hash password
    const hashedPassword = await bcrypt.hash(adminUser.password, 10);

    // Create admin user
    const pool = safeGetPool();
    if (!pool) {
      return res.status(500).json({ message: 'Database connection failed after initialization' });
    }
    await pool.execute(
      'INSERT INTO users (username, password, email) VALUES (?, ?, ?)',
      [adminUser.username, hashedPassword, adminUser.email || '']
    );

    // Initialize default settings
    const defaultSettings = {
      siteName: 'ویراوینگ',
      seo: { 
        title: 'ویراوینگ - طراحی سایت و اپلیکیشن', 
        description: 'ویراوینگ پیشرو در طراحی سایت، اپلیکیشن و راهکارهای هوشمند تحت وب. تیم حرفه‌ای ما آماده ارائه خدمات با کیفیت است.', 
        keywords: ['طراحی سایت', 'توسعه اپلیکیشن', 'سئو', 'طراحی UI/UX'],
        robots: 'index, follow',
        ogType: 'website',
        ogLocale: 'fa_IR',
        twitterCard: 'summary_large_image',
        schemaType: 'Organization'
      },
      hero: { badge: 'آینده وب اینجاست', title: 'ما رویاهای دیجیتال شما را', highlightedWord: 'خلق می‌کنیم', description: 'توسعه تخصصی وب و موبایل', ctaText: 'مشاوره رایگان' },
      footer: { socials: {}, contact: { address: 'تهران', phone: '021-88881234', email: 'info@viraving.com' } },
      navigation: {
        header: [
          { id: '1', name: 'خانه', href: '#home' },
          { id: '2', name: 'پروژه‌ها', href: '#portfolio' },
          { id: '3', name: 'وبلاگ', href: '#blog' },
          { id: '4', name: 'مشاوره هوشمند', href: '#home', anchor: 'ai-consultant' },
          { id: '5', name: 'تماس با ما', href: '#contact', special: true },
        ],
        footer: {
          quickAccess: [
            { id: 'f1', name: 'خانه', href: '#home' },
            { id: 'f2', name: 'نمونه کارها', href: '#portfolio' },
            { id: 'f3', name: 'مشاوره هوشمند', href: '#home', anchor: 'ai-consultant' },
            { id: 'f4', name: 'شروع همکاری', href: '#contact' },
          ],
          services: [
            { id: 's1', name: 'طراحی وب‌سایت', href: '#home', anchor: 'services' },
            { id: 's2', name: 'توسعه اپلیکیشن', href: '#home', anchor: 'services' },
            { id: 's3', name: 'طراحی UI/UX', href: '#home', anchor: 'services' },
            { id: 's4', name: 'سئو و مارکتینگ', href: '#home', anchor: 'services' },
          ]
        },
        mobile: [
          { id: 'm1', name: 'خانه', href: '#home', iconName: 'Home' },
          { id: 'm2', name: 'پروژه‌ها', href: '#portfolio', iconName: 'Briefcase' },
          { id: 'm3', name: 'مشاور AI', href: '#home', anchor: 'ai-consultant', iconName: 'Zap' },
          { id: 'm4', name: 'تماس', href: '#contact', iconName: 'Phone' },
        ]
      },
      aiConfig: { 
        enabled: true, 
        provider: 'liara',
        modelName: 'openai/gpt-5-mini:online',
        systemInstruction: 'شما یک مشاور فنی ارشد در شرکت نرم‌افزاری "ویراوینگ" هستید. وظیفه شما تحلیل ایده پروژه کاربر و پیشنهاد یک استک تکنولوژی مدرن و یک نقشه راه (Roadmap) کلی است. پاسخ باید کاملا به زبان فارسی باشد.',
        apiKey: ''
      },
      legal: {
        terms: '<h2>قوانین و مقررات</h2><p>استفاده از خدمات ویراوینگ به معنی پذیرش این قوانین است...</p>',
        privacy: '<h2>حریم خصوصی</h2><p>ما به حریم خصوصی شما احترام می‌گذاریم و اطلاعات شما نزد ما محفوظ است...</p>'
      }
    };

    await pool.execute(
      'INSERT INTO settings (data) VALUES (?)',
      [JSON.stringify(defaultSettings)]
    );

    res.json({ success: true });
  } catch (error) {
    logger.error('Installation error', { error: error.message, stack: error.stack });
    res.status(500).json({ message: error.message || 'Installation failed' });
  }
});

module.exports = router;
