const express = require('express');
const cors = require('cors');
const bodyParser = require('body-parser');
const path = require('path');
const fs = require('fs-extra');
const apiRoutes = require('./routes');
const { initDB } = require('./db');
const logger = require('./logger');

const app = express();
const PORT = process.env.PORT || 3001;

// Uploads directory (read-only /app in Liara, use disk mount like /storage/uploads)
const uploadsRoot =
  process.env.UPLOADS_DIR ||
  (process.env.NODE_ENV === 'production'
    ? '/storage/uploads'
    : path.join(__dirname, 'public/uploads'));

// Ensure uploads directory exists (no-op if read-only in production without disk)
try {
  fs.ensureDirSync(uploadsRoot);
} catch (e) {
  logger.warn('Could not ensure uploads directory (may be read-only in production)', { error: e.message });
}

// Initialize database (only if DB credentials are set)
if (process.env.DB_HOST && process.env.DB_NAME) {
  initDB()
    .then(() => {
      logger.info('Database initialized successfully');
    })
    .catch(err => {
      logger.error('Failed to initialize database', { error: err.message });
    // Don't exit - allow app to start (installer will handle DB setup)
  });
}

// Trust proxy (required for rate limiting behind reverse proxy like Liara)
// Set to 1 to trust only the first proxy (more secure than true)
app.set('trust proxy', 1);

// Middleware
app.use(cors());
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));

// API Routes
app.use('/api', apiRoutes);

// Static Files (Uploaded Images)
app.use('/uploads', express.static(uploadsRoot));

// Serve React Frontend (Production)
// Liara will build the frontend to the 'dist' folder in the root (parent of backend)
const distPath = path.join(__dirname, '../dist');

// Serve static assets from dist
app.use(express.static(distPath));

// Catch-all handler for React Router (SPA)
app.get('*', (req, res) => {
  // Don't intercept API calls or static uploads if they fall through
  if (req.url.startsWith('/api') || req.url.startsWith('/uploads')) {
      return res.status(404).send('Not found');
  }
  
  const indexPath = path.join(distPath, 'index.html');
  if (fs.existsSync(indexPath)) {
      res.sendFile(indexPath);
  } else {
      // If index.html doesn't exist (e.g. during dev without build), send a clear message
      res.status(404).send('Frontend build not found. Please run "npm run build" to generate the dist folder.');
  }
});

// Error handling middleware
app.use((err, req, res, next) => {
  logger.error('Unhandled error', { 
    error: err.message, 
    stack: err.stack,
    path: req.path,
    method: req.method
  });
  res.status(500).json({ message: 'Internal server error' });
});

// Start Server
app.listen(PORT, () => {
  logger.info(`Server is running on port ${PORT}`, { 
    env: process.env.NODE_ENV || 'development',
    port: PORT 
  });
});