import React from 'react';

interface ChartData {
  label: string;
  value: number;
  color?: string;
}

interface AdminChartProps {
  title: string;
  data: ChartData[];
}

const AdminChart: React.FC<AdminChartProps> = ({ title, data }) => {
  const maxValue = Math.max(...data.map(d => d.value));
  
  return (
    <div className="bg-white dark:bg-slate-800 p-6 rounded-2xl border border-slate-200 dark:border-slate-700 shadow-sm h-full flex flex-col">
      <h3 className="font-bold text-lg dark:text-white mb-6">{title}</h3>
      
      <div className="flex-1 flex items-end justify-between gap-2 sm:gap-4 min-h-[200px] pb-2 border-b border-slate-100 dark:border-slate-700">
        {data.map((item, index) => {
          const heightPercentage = maxValue > 0 ? (item.value / maxValue) * 100 : 0;
          const barColor = item.color || 'bg-indigo-500';
          
          return (
            <div key={index} className="flex flex-col items-center flex-1 group relative">
               {/* Tooltip */}
               <div className="absolute -top-10 opacity-0 group-hover:opacity-100 transition-opacity bg-slate-900 text-white text-xs py-1 px-2 rounded pointer-events-none mb-2 whitespace-nowrap z-10">
                 {item.label}: {item.value}
               </div>

               {/* Bar */}
               <div 
                 className={`w-full max-w-[40px] rounded-t-lg transition-all duration-1000 ease-out ${barColor} hover:opacity-80`}
                 style={{ height: `${Math.max(heightPercentage, 2)}%` }} // Min height 2% for visibility
               ></div>
            </div>
          );
        })}
      </div>
      
      {/* Labels */}
      <div className="flex justify-between gap-2 sm:gap-4 mt-3">
        {data.map((item, index) => (
           <div key={index} className="flex-1 text-center">
             <p className="text-[10px] sm:text-xs text-slate-500 truncate" title={item.label}>{item.label}</p>
           </div>
        ))}
      </div>
    </div>
  );
};

export default AdminChart;