import React, { useEffect, useState } from 'react';
import { ArrowRight, Calendar, Clock, User, Share2 } from 'lucide-react';
import { DataService, fireToast } from '../services/dataService';
import { BlogPost } from '../types';

interface BlogPostProps {
  id: string;
}

const BlogPostDetail: React.FC<BlogPostProps> = ({ id }) => {
  const [post, setPost] = useState<BlogPost | undefined>(undefined);
  const [readingProgress, setReadingProgress] = useState(0);

  useEffect(() => {
    const fetchData = async () => {
      const foundPost = await DataService.getPostById(id);
      setPost(foundPost);
      window.scrollTo(0, 0);
    };
    fetchData();
  }, [id]);

  // Handle Scroll Progress
  useEffect(() => {
    const handleScroll = () => {
      const totalHeight = document.body.scrollHeight - window.innerHeight;
      const progress = (window.scrollY / totalHeight) * 100;
      setReadingProgress(progress);
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  // SEO Update
  useEffect(() => {
    if (post) {
      if (post.seo?.metaTitle) {
        document.title = post.seo.metaTitle;
      } else {
        document.title = post.title;
      }
      
      if (post.seo?.metaDescription) {
         let metaDesc = document.querySelector('meta[name="description"]');
         if (!metaDesc) {
           metaDesc = document.createElement('meta');
           metaDesc.setAttribute('name', 'description');
           document.head.appendChild(metaDesc);
         }
         metaDesc.setAttribute('content', post.seo.metaDescription);
      } else {
         let metaDesc = document.querySelector('meta[name="description"]');
         if (metaDesc) {
            metaDesc.setAttribute('content', post.excerpt);
         }
      }
    }
  }, [post]);

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  const handleShare = () => {
    navigator.clipboard.writeText(window.location.href);
    fireToast('لینک مقاله کپی شد', 'success');
  };

  if (!post) {
    return (
      <div className="min-h-screen pt-32 flex flex-col items-center justify-center text-slate-500 dark:text-slate-400">
        <h2 className="text-2xl font-bold mb-4">مقاله مورد نظر پیدا نشد</h2>
        <a href="#blog" onClick={(e) => handleNavClick(e, '#blog')} className="text-indigo-600 hover:underline">بازگشت به وبلاگ</a>
      </div>
    );
  }

  return (
    <article className="min-h-screen pt-24 md:pt-32 pb-24 bg-slate-50 dark:bg-slate-950 transition-colors duration-300 relative">
      
      {/* Reading Progress Bar */}
      <div className="fixed top-0 left-0 h-1 bg-indigo-600 z-[60] transition-all duration-100" style={{ width: `${readingProgress}%` }}></div>

      {/* Header Image */}
      <div className="container mx-auto px-4 mb-8">
        <a href="#blog" onClick={(e) => handleNavClick(e, '#blog')} className="inline-flex items-center gap-2 text-slate-500 hover:text-indigo-600 mb-6 transition-colors">
          <ArrowRight size={18} /> بازگشت به مقالات
        </a>
        
        <div className="aspect-[21/9] w-full rounded-3xl overflow-hidden shadow-2xl relative">
          <img src={post.image} alt={post.title} className="w-full h-full object-cover" />
          <div className="absolute inset-0 bg-gradient-to-t from-slate-900/80 to-transparent"></div>
          <div className="absolute bottom-0 right-0 p-6 md:p-12 w-full">
             <h1 className="text-2xl md:text-5xl font-black text-white mb-6 leading-tight drop-shadow-md">{post.title}</h1>
             <div className="flex flex-wrap items-center gap-6 text-slate-200 text-sm md:text-base">
                <span className="flex items-center gap-2"><User size={18} /> {post.author}</span>
                <span className="flex items-center gap-2"><Calendar size={18} /> {post.date}</span>
                <span className="flex items-center gap-2"><Clock size={18} /> {post.readTime}</span>
             </div>
          </div>
        </div>
      </div>

      <div className="container mx-auto px-4 grid lg:grid-cols-4 gap-12">
        
        {/* Main Content */}
        <div className="lg:col-span-3">
          <div className="bg-white dark:bg-slate-900 rounded-3xl p-8 border border-slate-200 dark:border-slate-800 shadow-xl">
             <div 
               className="prose prose-lg dark:prose-invert max-w-none prose-headings:font-bold prose-headings:text-slate-900 dark:prose-headings:text-white prose-p:text-slate-600 dark:prose-p:text-slate-300 prose-a:text-indigo-600 dark:prose-a:text-indigo-400 prose-img:rounded-2xl"
               dangerouslySetInnerHTML={{ __html: post.content }} 
             />
          </div>
        </div>

        {/* Sidebar */}
        <div className="space-y-6">
           <div className="bg-white dark:bg-slate-900 rounded-3xl p-6 border border-slate-200 dark:border-slate-800 shadow-lg sticky top-24">
              <h3 className="font-bold text-slate-900 dark:text-white mb-4">اشتراک‌گذاری</h3>
              <div className="flex gap-4 items-center">
                 <button onClick={handleShare} className="w-10 h-10 rounded-full bg-slate-100 dark:bg-slate-800 flex items-center justify-center text-slate-600 dark:text-slate-400 hover:bg-indigo-100 hover:text-indigo-600 transition-colors">
                   <Share2 size={18} />
                 </button>
                 <span className="text-xs text-slate-400">برای کپی لینک کلیک کنید</span>
              </div>
           </div>
        </div>

      </div>
    </article>
  );
};

export default BlogPostDetail;