import React, { useState, useEffect } from 'react';
import { Facebook, Twitter, Instagram, Linkedin, Mail, Phone, MapPin, ChevronDown, ExternalLink } from 'lucide-react';
import { DataService } from '../services/dataService';
import { AppSettings, MenuItem } from '../types';

interface FooterSectionProps {
  title: string;
  children: React.ReactNode;
  defaultOpen?: boolean;
}

const FooterSection: React.FC<FooterSectionProps> = ({ title, children, defaultOpen = false }) => {
  const [isOpen, setIsOpen] = useState(defaultOpen);

  return (
    <div className="border-b border-slate-200 dark:border-slate-800/50 md:border-none last:border-none">
      <button 
        onClick={() => setIsOpen(!isOpen)}
        className="w-full flex items-center justify-between py-5 md:py-0 md:mb-6 group"
      >
        <h4 className="text-slate-900 dark:text-white font-bold text-lg md:text-base group-hover:text-indigo-600 dark:group-hover:text-indigo-400 transition-colors flex items-center gap-2">
           <span className="w-1.5 h-1.5 rounded-full bg-indigo-500 md:hidden"></span>
           {title}
        </h4>
        <ChevronDown 
          className={`md:hidden text-slate-500 transition-transform duration-300 ${isOpen ? 'rotate-180 text-indigo-500' : ''}`} 
          size={20} 
        />
      </button>
      <div className={`overflow-hidden transition-all duration-300 ease-in-out md:block ${isOpen ? 'max-h-96 opacity-100 mb-6' : 'max-h-0 opacity-0 md:max-h-full md:opacity-100 md:mb-0'}`}>
        {children}
      </div>
    </div>
  );
};

const Footer: React.FC = () => {
  const [settings, setSettings] = useState<AppSettings | null>(null);

  useEffect(() => {
    const update = async () => {
      const data = await DataService.getSettings();
      setSettings(data);
    };
    update();
    window.addEventListener('settings-updated', update);
    return () => window.removeEventListener('settings-updated', update);
  }, []);

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  if (!settings) return null;

  const { socials, contact } = settings.footer;
  const { footer: footerNav } = settings.navigation || { quickAccess: [], services: [] };

  return (
    <footer className="bg-white dark:bg-slate-950 pt-12 pb-24 md:pb-12 border-t border-slate-200 dark:border-slate-900 relative transition-colors duration-300">
      {/* Background Gradient */}
      <div className="absolute top-0 left-0 w-full h-full overflow-hidden pointer-events-none z-0">
          <div className="absolute bottom-0 right-0 w-[500px] h-[500px] bg-indigo-100/50 dark:bg-indigo-900/10 rounded-full blur-[100px]"></div>
      </div>

      <div className="container mx-auto px-6 relative z-10">
        
        {/* Top Section: Brand info */}
        <div className="flex flex-col md:flex-row justify-between items-start gap-8 mb-8 md:mb-12">
           <div className="space-y-4 max-w-sm">
             <div className="flex items-center gap-3">
                <div className="w-10 h-10 bg-gradient-to-br from-indigo-600 to-slate-800 rounded-xl flex items-center justify-center shadow-lg shadow-indigo-500/20 ring-1 ring-white/10">
                   {settings.logoUrl ? (
                      <img src={settings.logoUrl} alt="Logo" className="w-full h-full object-cover rounded-xl" />
                   ) : (
                      <span className="text-white font-black text-xl">V</span>
                   )}
                </div>
                <span className="text-2xl font-bold text-slate-900 dark:text-white tracking-wide">{settings.siteName}</span>
             </div>
             <p className="text-slate-600 dark:text-slate-400 leading-relaxed text-sm text-justify">
               ما در {settings.siteName} با ترکیب خلاقیت و تکنولوژی، راهکارهای دیجیتال هوشمند خلق می‌کنیم تا کسب‌وکار شما در دنیای مدرن، متفاوت و پیشرو باشد.
             </p>
           </div>
           
           {/* Desktop Socials */}
           <div className="hidden md:flex gap-3">
              {socials.instagram && (
                <a href={socials.instagram} className="w-10 h-10 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white hover:border-indigo-500 transition-all duration-300 group shadow-lg">
                  <Instagram size={18} className="group-hover:scale-110 transition-transform" />
                </a>
              )}
              {socials.twitter && (
                <a href={socials.twitter} className="w-10 h-10 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white hover:border-indigo-500 transition-all duration-300 group shadow-lg">
                  <Twitter size={18} className="group-hover:scale-110 transition-transform" />
                </a>
              )}
              {socials.linkedin && (
                <a href={socials.linkedin} className="w-10 h-10 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white hover:border-indigo-500 transition-all duration-300 group shadow-lg">
                  <Linkedin size={18} className="group-hover:scale-110 transition-transform" />
                </a>
              )}
              {socials.facebook && (
                <a href={socials.facebook} className="w-10 h-10 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white hover:border-indigo-500 transition-all duration-300 group shadow-lg">
                  <Facebook size={18} className="group-hover:scale-110 transition-transform" />
                </a>
              )}
           </div>
        </div>

        {/* App-like Accordion Sections */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-0 md:gap-12 mb-10 border-t border-b border-slate-200 dark:border-slate-800/50 md:border-none pt-2 md:pt-0">
          
          <FooterSection title="دسترسی سریع">
             <ul className="space-y-3 text-sm text-slate-600 dark:text-slate-400">
               {footerNav?.quickAccess?.map((link) => {
                 const target = link.anchor ? `${link.href}?scroll=${link.anchor}` : link.href;
                 return (
                   <li key={link.id}>
                     <a href={target} onClick={(e) => handleNavClick(e, target)} className="flex items-center gap-2 hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors cursor-pointer py-1">
                       <ExternalLink size={14} className="opacity-50" /> {link.name}
                     </a>
                   </li>
                 );
               })}
             </ul>
          </FooterSection>

          <FooterSection title="خدمات ما">
             <ul className="space-y-3 text-sm text-slate-600 dark:text-slate-400">
               {footerNav?.services?.map((link) => {
                 const target = link.anchor ? `${link.href}?scroll=${link.anchor}` : link.href;
                 return (
                   <li key={link.id}>
                     <a href={target} onClick={(e) => handleNavClick(e, target)} className="hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors cursor-pointer block py-1">
                       {link.name}
                     </a>
                   </li>
                 );
               })}
             </ul>
          </FooterSection>

          <FooterSection title="تماس با ما" defaultOpen={false}>
             <ul className="space-y-4 text-sm text-slate-600 dark:text-slate-400">
                <li className="flex items-start gap-3 bg-slate-50 dark:bg-slate-900/50 p-3 rounded-xl border border-slate-200 dark:border-slate-800/50">
                  <MapPin size={18} className="text-indigo-500 shrink-0 mt-0.5" />
                  <span className="leading-relaxed">{contact.address}</span>
                </li>
                <li className="flex items-center gap-3 bg-slate-50 dark:bg-slate-900/50 p-3 rounded-xl border border-slate-200 dark:border-slate-800/50">
                  <Phone size={18} className="text-green-600 dark:text-green-500 shrink-0" />
                  <span className="font-mono dir-ltr">{contact.phone}</span>
                </li>
                <li className="flex items-center gap-3 bg-slate-50 dark:bg-slate-900/50 p-3 rounded-xl border border-slate-200 dark:border-slate-800/50">
                  <Mail size={18} className="text-pink-600 dark:text-pink-500 shrink-0" />
                  <span className="font-mono">{contact.email}</span>
                </li>
             </ul>
          </FooterSection>

        </div>

        {/* Mobile Socials (Centered & Large Touch Targets) */}
        <div className="md:hidden flex justify-center gap-4 mb-8">
            {socials.instagram && (
              <a href={socials.instagram} className="w-12 h-12 rounded-2xl bg-white dark:bg-slate-800/80 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white transition-all shadow-lg active:scale-95">
                <Instagram size={22} />
              </a>
            )}
            {socials.twitter && (
              <a href={socials.twitter} className="w-12 h-12 rounded-2xl bg-white dark:bg-slate-800/80 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white transition-all shadow-lg active:scale-95">
                <Twitter size={22} />
              </a>
            )}
            {socials.linkedin && (
              <a href={socials.linkedin} className="w-12 h-12 rounded-2xl bg-white dark:bg-slate-800/80 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white transition-all shadow-lg active:scale-95">
                <Linkedin size={22} />
              </a>
            )}
            {socials.facebook && (
              <a href={socials.facebook} className="w-12 h-12 rounded-2xl bg-white dark:bg-slate-800/80 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-slate-500 dark:text-slate-400 hover:bg-indigo-600 hover:text-white transition-all shadow-lg active:scale-95">
                <Facebook size={22} />
              </a>
            )}
        </div>

        {/* Copyright & Legal */}
        <div className="border-t border-slate-200 dark:border-slate-800 pt-8 flex flex-col md:flex-row justify-between items-center gap-4">
           <p className="text-slate-500 text-xs md:text-sm text-center md:text-right">
             © ۱۴۰۳ تمامی حقوق محفوظ است | توسعه با ❤️ در <span className="text-indigo-600 dark:text-indigo-400 font-bold">{settings.siteName}</span>
           </p>
           <div className="flex gap-6 text-xs md:text-sm text-slate-500">
              <a href="#admin" onClick={(e) => handleNavClick(e, '#admin')} className="hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors">مدیریت</a>
              <a href="#terms" onClick={(e) => handleNavClick(e, '#terms')} className="hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors">قوانین</a>
              <a href="#privacy" onClick={(e) => handleNavClick(e, '#privacy')} className="hover:text-indigo-600 dark:hover:text-indigo-400 transition-colors">حریم خصوصی</a>
           </div>
        </div>

      </div>
    </footer>
  );
};

export default Footer;