import React, { useState, useEffect } from 'react';
import { ArrowLeft, Code, Layers, Zap } from 'lucide-react';
import { DataService } from '../services/dataService';
import { AppSettings } from '../types';

const Hero: React.FC = () => {
  const [settings, setSettings] = useState<AppSettings | null>(null);

  useEffect(() => {
    const update = async () => {
      const data = await DataService.getSettings();
      setSettings(data);
    };
    update();
    window.addEventListener('settings-updated', update);
    return () => window.removeEventListener('settings-updated', update);
  }, []);

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  if (!settings) {
    return null; // Or loading spinner
  }

  const { hero } = settings;

  return (
    <section id="hero" className="relative min-h-screen flex items-center justify-center pt-20 md:pt-32 overflow-hidden bg-slate-50 dark:bg-slate-900 transition-colors duration-300">
      
      {/* Background Elements */}
      <div className="absolute top-0 left-0 w-full h-full overflow-hidden z-0">
        <div className="absolute top-[-10%] left-[-10%] w-[500px] h-[500px] bg-indigo-600/10 dark:bg-indigo-700/20 rounded-full blur-[120px] mix-blend-multiply dark:mix-blend-screen opacity-50 animate-pulse"></div>
        <div className="absolute bottom-[-10%] right-[-10%] w-[500px] h-[500px] bg-slate-500/10 dark:bg-slate-600/20 rounded-full blur-[120px] mix-blend-multiply dark:mix-blend-screen opacity-50"></div>
        <div className="absolute top-[40%] left-[60%] w-[300px] h-[300px] bg-teal-400/10 dark:bg-teal-500/10 rounded-full blur-[100px] mix-blend-multiply dark:mix-blend-screen opacity-30"></div>
      </div>

      <div className="container mx-auto px-4 z-10 grid md:grid-cols-2 gap-12 items-center">
        
        {/* Text Content */}
        <div className="text-right space-y-6">
          <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-white/60 dark:bg-slate-800/50 border border-slate-200 dark:border-slate-700 text-indigo-600 dark:text-indigo-400 text-sm font-medium shadow-sm">
            <span className="relative flex h-2 w-2">
              <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-indigo-400 opacity-75"></span>
              <span className="relative inline-flex rounded-full h-2 w-2 bg-indigo-500"></span>
            </span>
            {hero?.badge || 'آینده وب اینجاست'}
          </div>
          
          <h1 className="text-4xl md:text-6xl lg:text-[3rem] lg:leading-[1.5] font-black text-slate-900 dark:text-white leading-tight">
            {hero?.title || 'ما رویاهای دیجیتال شما را'} <br/>
            <span className="text-transparent bg-clip-text bg-gradient-to-r from-indigo-600 via-teal-500 to-slate-500 dark:from-indigo-400 dark:via-teal-400 dark:to-slate-400">
              {hero?.highlightedWord || 'خلق می‌کنیم'}
            </span>
          </h1>
          
          <p className="text-lg md:text-xl text-slate-600 dark:text-slate-400 max-w-xl leading-relaxed">
            {hero?.description || 'ویراوینگ با ترکیب هنر طراحی و قدرت تکنولوژی، محصولاتی می‌سازد که کسب‌وکـار شما را در دنیای دیجیتال متمایز می‌کند.'}
          </p>
          
          <div className="flex flex-col sm:flex-row gap-4 pt-4">
            <a 
              href="#home?scroll=ai-consultant" 
              onClick={(e) => handleNavClick(e, '#home?scroll=ai-consultant')}
              className="group inline-flex items-center justify-center px-8 py-4 text-base font-bold text-white bg-indigo-600 rounded-xl hover:bg-indigo-700 transition-all duration-300 shadow-lg shadow-indigo-500/30 hover:shadow-indigo-500/50 cursor-pointer"
            >
              {hero?.ctaText || 'مشاوره رایگان هوشمند'}
              <Zap className="mr-2 w-5 h-5 group-hover:text-yellow-300 transition-colors" />
            </a>
            <a 
              href="#portfolio" 
              onClick={(e) => handleNavClick(e, '#portfolio')}
              className="inline-flex items-center justify-center px-8 py-4 text-base font-medium text-slate-600 dark:text-slate-300 bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 rounded-xl hover:bg-slate-50 dark:hover:bg-slate-750 hover:text-indigo-600 dark:hover:text-white transition-all cursor-pointer shadow-sm hover:shadow"
            >
              دیدن نمونه کارها
              <ArrowLeft className="mr-2 w-5 h-5" />
            </a>
          </div>
          
          <div className="pt-8 flex items-center gap-8 text-slate-500 text-sm">
            <div className="flex items-center gap-2">
              <Code size={18} className="text-teal-600 dark:text-teal-500" />
              <span>کدنویسی تمیز</span>
            </div>
            <div className="flex items-center gap-2">
              <Layers size={18} className="text-indigo-600 dark:text-indigo-500" />
              <span>طراحی مدرن</span>
            </div>
          </div>
        </div>

        {/* Visual Content (Abstract UI Mockup) */}
        <div className="relative hidden md:block">
           <div className="relative z-10 bg-white/80 dark:bg-slate-800/50 border border-slate-200 dark:border-slate-700 backdrop-blur-xl rounded-2xl p-6 shadow-2xl transform rotate-[-3deg] hover:rotate-0 transition-transform duration-500">
              <div className="flex items-center gap-2 mb-4">
                <div className="w-3 h-3 rounded-full bg-red-500"></div>
                <div className="w-3 h-3 rounded-full bg-yellow-500"></div>
                <div className="w-3 h-3 rounded-full bg-green-500"></div>
              </div>
              <div className="space-y-4">
                 <div className="h-40 bg-gradient-to-r from-slate-200 to-slate-100 dark:from-slate-700 dark:to-slate-600 rounded-xl animate-pulse"></div>
                 <div className="grid grid-cols-3 gap-4">
                    <div className="h-24 bg-slate-100 dark:bg-slate-700/50 rounded-xl"></div>
                    <div className="h-24 bg-slate-100 dark:bg-slate-700/50 rounded-xl"></div>
                    <div className="h-24 bg-slate-100 dark:bg-slate-700/50 rounded-xl"></div>
                 </div>
                 <div className="h-8 w-1/2 bg-slate-100 dark:bg-slate-700/30 rounded-lg"></div>
              </div>
           </div>
           
           <div className="absolute -bottom-10 -right-10 w-full h-full bg-gradient-to-tr from-indigo-500 to-teal-600 rounded-2xl opacity-10 dark:opacity-20 z-0 transform rotate-[6deg]"></div>
        </div>
      </div>
    </section>
  );
};

export default Hero;