import React, { useState } from 'react';
import { Database, User, Check, ArrowLeft, Server, Lock, Shield } from 'lucide-react';
import { DataService } from '../services/dataService';
import { AdminUser, DBConfig } from '../types';

const Installer: React.FC = () => {
  const [step, setStep] = useState(1);
  const [loading, setLoading] = useState(false);
  const [dbConfig, setDbConfig] = useState<DBConfig>({
    host: 'localhost',
    port: '3306',
    name: 'viraving_db',
    user: 'root',
    pass: '',
    type: 'mysql'
  });
  const [adminUser, setAdminUser] = useState<AdminUser>({
    username: 'admin',
    password: '',
    email: 'admin@viraving.com'
  });
  const [confirmPass, setConfirmPass] = useState('');
  const [error, setError] = useState('');

  const handleDbSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    setError('');
    
    if (!dbConfig.host || !dbConfig.name || !dbConfig.port || !dbConfig.user) {
      setError('لطفا اطلاعات دیتابیس را کامل وارد کنید');
      setLoading(false);
      return;
    }

    // Try to test database connection via API
    try {
      // This will be handled by the install endpoint which tests the connection
      // For now, just validate and move to next step
      setError('');
      setStep(2);
    } catch (err: any) {
      setError('خطا در اتصال به دیتابیس: ' + (err.message || 'لطفا اطلاعات را بررسی کنید'));
    } finally {
      setLoading(false);
    }
  };

  const handleAdminSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!adminUser.username || !adminUser.password) {
      setError('نام کاربری و رمز عبور الزامی است');
      return;
    }
    if (adminUser.password !== confirmPass) {
      setError('تکرار رمز عبور مطابقت ندارد');
      return;
    }

    setLoading(true);
    try {
      await DataService.installApp(dbConfig, adminUser);
      // No need to set loading false as the app will reload
    } catch (err) {
      setError('خطا در نصب برنامه');
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen bg-slate-900 flex items-center justify-center p-4 dir-rtl font-vazir">
      {/* Background Effect */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
         <div className="absolute top-[-10%] left-[-10%] w-[500px] h-[500px] bg-indigo-600/20 rounded-full blur-[120px]"></div>
         <div className="absolute bottom-[-10%] right-[-10%] w-[500px] h-[500px] bg-teal-500/10 rounded-full blur-[120px]"></div>
      </div>

      <div className="bg-slate-800/80 backdrop-blur-xl border border-slate-700 w-full max-w-lg rounded-3xl shadow-2xl overflow-hidden relative z-10">
        
        {/* Header */}
        <div className="bg-slate-900/50 p-6 border-b border-slate-700 flex items-center justify-between">
          <div className="flex items-center gap-3">
             <div className="w-10 h-10 bg-indigo-600 rounded-lg flex items-center justify-center text-white">
                <Server size={20} />
             </div>
             <div>
                <h1 className="text-white font-bold text-lg">نصب‌کننده ویراوینگ</h1>
                <p className="text-slate-400 text-xs">نسخه ۱.۰.۰</p>
             </div>
          </div>
          <div className="flex gap-2">
            <div className={`w-3 h-3 rounded-full ${step >= 1 ? 'bg-indigo-500' : 'bg-slate-700'}`}></div>
            <div className={`w-3 h-3 rounded-full ${step >= 2 ? 'bg-indigo-500' : 'bg-slate-700'}`}></div>
          </div>
        </div>

        {/* Content */}
        <div className="p-8">
           
           {error && (
             <div className="bg-red-500/10 border border-red-500/50 text-red-400 p-3 rounded-xl text-sm mb-6 text-center">
               {error}
             </div>
           )}

           {step === 1 && (
             <form onSubmit={handleDbSubmit} className="space-y-4 animate-fade-in">
                <div className="text-center mb-6">
                   <div className="w-16 h-16 bg-slate-700/50 rounded-full flex items-center justify-center mx-auto mb-4 text-indigo-400 border border-slate-600">
                     <Database size={32} />
                   </div>
                   <h2 className="text-white font-bold text-xl">اتصال به دیتابیس (MySQL)</h2>
                   <p className="text-slate-400 text-sm mt-2">اطلاعات اتصال به سرور دیتابیس را وارد کنید</p>
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-slate-400 text-xs mb-1">هاست (Host)</label>
                    <input type="text" dir="ltr" value={dbConfig.host} onChange={e => setDbConfig({...dbConfig, host: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-2 text-white focus:border-indigo-500 outline-none" placeholder="localhost" />
                  </div>
                  <div>
                    <label className="block text-slate-400 text-xs mb-1">پورت (Port)</label>
                    <input type="text" dir="ltr" value={dbConfig.port} onChange={e => setDbConfig({...dbConfig, port: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-2 text-white focus:border-indigo-500 outline-none" placeholder="3306" />
                  </div>
                </div>
                
                <div>
                    <label className="block text-slate-400 text-xs mb-1">نام دیتابیس</label>
                    <input type="text" dir="ltr" value={dbConfig.name} onChange={e => setDbConfig({...dbConfig, name: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-2 text-white focus:border-indigo-500 outline-none" />
                </div>
                
                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-slate-400 text-xs mb-1">نام کاربری (User)</label>
                    <input type="text" dir="ltr" value={dbConfig.user} onChange={e => setDbConfig({...dbConfig, user: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-2 text-white focus:border-indigo-500 outline-none" />
                  </div>
                  <div>
                    <label className="block text-slate-400 text-xs mb-1">رمز عبور (Pass)</label>
                    <input type="password" dir="ltr" value={dbConfig.pass} onChange={e => setDbConfig({...dbConfig, pass: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-2 text-white focus:border-indigo-500 outline-none" />
                  </div>
                </div>

                <button type="submit" disabled={loading} className="w-full bg-indigo-600 hover:bg-indigo-500 text-white font-bold py-3 rounded-xl transition-all mt-4 flex items-center justify-center gap-2">
                  {loading ? 'در حال بررسی اتصال...' : (
                    <>
                      <span>مرحله بعد</span>
                      <ArrowLeft size={18} />
                    </>
                  )}
                </button>
             </form>
           )}

           {step === 2 && (
             <form onSubmit={handleAdminSubmit} className="space-y-4 animate-fade-in">
                <div className="text-center mb-6">
                   <div className="w-16 h-16 bg-slate-700/50 rounded-full flex items-center justify-center mx-auto mb-4 text-green-400 border border-slate-600">
                     <Shield size={32} />
                   </div>
                   <h2 className="text-white font-bold text-xl">ایجاد مدیر کل</h2>
                   <p className="text-slate-400 text-sm mt-2">اطلاعات ورود به پنل مدیریت را تعیین کنید</p>
                </div>

                <div>
                  <label className="block text-slate-400 text-xs mb-1">نام کاربری</label>
                  <div className="relative">
                    <User size={16} className="absolute top-3 right-3 text-slate-500" />
                    <input type="text" dir="ltr" value={adminUser.username} onChange={e => setAdminUser({...adminUser, username: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-10 py-2 text-white focus:border-indigo-500 outline-none" />
                  </div>
                </div>

                <div>
                  <label className="block text-slate-400 text-xs mb-1">ایمیل</label>
                  <input type="email" dir="ltr" value={adminUser.email} onChange={e => setAdminUser({...adminUser, email: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-2 text-white focus:border-indigo-500 outline-none" />
                </div>

                <div className="grid grid-cols-2 gap-4">
                  <div>
                    <label className="block text-slate-400 text-xs mb-1">رمز عبور</label>
                    <div className="relative">
                       <Lock size={16} className="absolute top-3 right-3 text-slate-500" />
                       <input type="password" dir="ltr" value={adminUser.password} onChange={e => setAdminUser({...adminUser, password: e.target.value})} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-10 py-2 text-white focus:border-indigo-500 outline-none" />
                    </div>
                  </div>
                  <div>
                    <label className="block text-slate-400 text-xs mb-1">تکرار رمز</label>
                    <input type="password" dir="ltr" value={confirmPass} onChange={e => setConfirmPass(e.target.value)} className="w-full bg-slate-900 border border-slate-700 rounded-xl px-4 py-2 text-white focus:border-indigo-500 outline-none" />
                  </div>
                </div>

                <button type="submit" disabled={loading} className="w-full bg-green-600 hover:bg-green-500 text-white font-bold py-3 rounded-xl transition-all mt-4 flex items-center justify-center gap-2 shadow-lg shadow-green-900/20">
                  {loading ? 'در حال نصب و راه اندازی...' : (
                    <>
                      <Check size={18} />
                      <span>پایان نصب و ورود</span>
                    </>
                  )}
                </button>
             </form>
           )}

        </div>
        
        {/* Footer */}
        <div className="bg-slate-900/30 p-4 border-t border-slate-700 text-center">
          <p className="text-slate-500 text-xs">Viraving System Installer &copy; 2024</p>
        </div>
      </div>
    </div>
  );
};

export default Installer;