import React, { useEffect, useState } from 'react';
import { ArrowLeft, Loader2, PlusCircle, Search } from 'lucide-react';
import { DataService } from '../services/dataService';
import { Project } from '../types';
import { ProjectSkeleton } from './Skeletons';

const Portfolio: React.FC = () => {
  const [projectsData, setProjectsData] = useState<Project[]>([]);
  const [loading, setLoading] = useState(true);
  const [loadingMore, setLoadingMore] = useState(false);
  const [categories, setCategories] = useState<string[]>([]);
  const [activeCategory, setActiveCategory] = useState('All');
  const [searchTerm, setSearchTerm] = useState('');
  const [page, setPage] = useState(1);
  const [hasMore, setHasMore] = useState(true);
  const LIMIT = 6;

  const fetchProjects = async (reset: boolean = false) => {
    const targetPage = reset ? 1 : page;
    if (reset) {
        setLoading(true);
        setProjectsData([]);
    } else {
        setLoadingMore(true);
    }

    // Simulate slight network delay to show off the skeleton (remove in prod)
    if(reset) await new Promise(r => setTimeout(r, 800));

    const response = await DataService.getProjects(targetPage, LIMIT, searchTerm, activeCategory);
    
    if (reset) {
        setProjectsData(response.data);
    } else {
        setProjectsData(prev => [...prev, ...response.data]);
    }

    if (response.data.length < LIMIT || (reset ? response.data.length : projectsData.length + response.data.length) >= response.total) {
        setHasMore(false);
    } else {
        setHasMore(true);
    }

    if (reset) setPage(1);
    setLoading(false);
    setLoadingMore(false);
  };

  useEffect(() => {
    // Initial fetch and categories setup
    const init = async () => {
        const cats = ['All', 'Web App', 'Mobile App', 'Corporate', 'E-Commerce', 'Fintech']; 
        setCategories(cats);
        await fetchProjects(true);
    };
    init();
    
    // Listen for data updates
    const handleUpdate = () => fetchProjects(true);
    window.addEventListener('data-updated', handleUpdate);
    return () => window.removeEventListener('data-updated', handleUpdate);
  }, []);

  // Trigger fetch when category or search changes (debounce search)
  useEffect(() => {
      const delayDebounceFn = setTimeout(() => {
        fetchProjects(true);
      }, 500);

      return () => clearTimeout(delayDebounceFn);
  }, [searchTerm, activeCategory]);

  const loadMore = async () => {
    if (!hasMore || loadingMore) return;
    const nextPage = page + 1;
    setPage(nextPage); 
    
    setLoadingMore(true);
    const response = await DataService.getProjects(nextPage, LIMIT, searchTerm, activeCategory);
    
    setProjectsData(prev => [...prev, ...response.data]);
    
    if (response.data.length < LIMIT || projectsData.length + response.data.length >= response.total) {
        setHasMore(false);
    }
    setLoadingMore(false);
  };

  const handleNavClick = (e: React.MouseEvent<HTMLAnchorElement>, target: string) => {
    e.preventDefault();
    window.location.hash = target;
  };

  return (
    <section id="portfolio-page" className="min-h-screen pt-20 md:pt-32 pb-24 bg-slate-50 dark:bg-slate-950 animate-fade-in transition-colors duration-300">
      <div className="container mx-auto px-4">
        <div className="text-center mb-8 mt-8">
           <span className="text-indigo-600 dark:text-indigo-400 font-bold tracking-wider text-sm uppercase">Portfolio</span>
           <h2 className="text-3xl md:text-5xl font-bold text-slate-900 dark:text-white mt-2 mb-4">آرشیو پروژه‌ها</h2>
           <p className="text-slate-600 dark:text-slate-400 max-w-xl mx-auto">
             بررسی دقیق‌تر پروژه‌هایی که با چالش‌های فنی پیچیده و طراحی‌های خلاقانه اجرا شده‌اند.
           </p>
        </div>

        {/* Search & Filter Controls */}
        <div className="mb-10 max-w-4xl mx-auto">
            
            {/* Search Bar */}
            <div className="relative mb-6">
                <Search className="absolute right-4 top-3.5 text-slate-400" size={20} />
                <input 
                    type="text" 
                    placeholder="جستجو در نام یا توضیحات پروژه..." 
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="w-full bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-2xl px-12 py-3 outline-none focus:border-indigo-500 focus:ring-1 focus:ring-indigo-500 transition-all text-slate-700 dark:text-slate-200"
                />
            </div>

            {/* Categories */}
            <div className="flex flex-wrap justify-center gap-2">
                {categories.map((cat) => (
                <button
                    key={cat}
                    onClick={() => setActiveCategory(cat)}
                    className={`px-4 py-2 rounded-full text-sm font-medium transition-all ${
                    activeCategory === cat
                        ? 'bg-indigo-600 text-white shadow-lg shadow-indigo-500/30'
                        : 'bg-white dark:bg-slate-900 text-slate-600 dark:text-slate-400 hover:bg-indigo-50 dark:hover:bg-slate-800 border border-slate-100 dark:border-slate-800'
                    }`}
                >
                    {cat === 'All' ? 'همه موارد' : cat}
                </button>
                ))}
            </div>
        </div>

        {loading ? (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
             {Array.from({ length: 6 }).map((_, i) => (
               <ProjectSkeleton key={i} />
             ))}
          </div>
        ) : (
          <>
            {projectsData.length === 0 ? (
                <div className="text-center py-20 text-slate-500 dark:text-slate-400">
                    موردی یافت نشد.
                </div>
            ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
                {projectsData.map((project) => (
                    <a 
                    href={`#project/${project.id}`}
                    onClick={(e) => handleNavClick(e, `#project/${project.id}`)}
                    key={project.id} 
                    className="group relative overflow-hidden rounded-3xl cursor-pointer bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 hover:border-indigo-500/50 transition-all duration-300 hover:shadow-2xl hover:shadow-indigo-900/10 dark:hover:shadow-indigo-900/20 animate-fade-in-up"
                    >
                    <div className="aspect-[4/3] w-full overflow-hidden">
                        <img 
                        src={project.image} 
                        alt={project.title} 
                        className="w-full h-full object-cover transition-transform duration-700 group-hover:scale-110 opacity-90 group-hover:opacity-100"
                        />
                    </div>
                    
                    <div className="absolute inset-0 bg-gradient-to-t from-slate-900/90 via-slate-900/60 to-transparent opacity-90 transition-opacity duration-300"></div>

                    <div className="absolute bottom-0 left-0 right-0 p-6 transform translate-y-2 group-hover:translate-y-0 transition-transform duration-300">
                        <div className="flex justify-between items-end">
                        <div>
                            <span className="text-indigo-300 text-xs font-bold tracking-wider uppercase mb-2 inline-block bg-indigo-500/20 px-2 py-1 rounded">
                            {project.category}
                            </span>
                            <h3 className="text-xl font-bold text-white mb-1">
                            {project.title}
                            </h3>
                        </div>
                        <div className="w-10 h-10 bg-indigo-600 rounded-full flex items-center justify-center opacity-0 group-hover:opacity-100 transition-all duration-300 transform translate-x-4 group-hover:translate-x-0 shadow-lg">
                            <ArrowLeft size={20} className="text-white" />
                        </div>
                        </div>
                    </div>
                    </a>
                ))}
                </div>
            )}

            {/* Load More Button */}
            {hasMore && (
              <div className="flex justify-center mt-12">
                <button 
                  onClick={loadMore} 
                  disabled={loadingMore}
                  className="px-8 py-3 bg-white dark:bg-slate-900 border border-slate-200 dark:border-slate-800 rounded-xl text-slate-700 dark:text-slate-300 font-bold hover:bg-slate-50 dark:hover:bg-slate-800 transition-colors flex items-center gap-2 shadow-sm disabled:opacity-50"
                >
                  {loadingMore ? <Loader2 className="animate-spin" /> : <PlusCircle size={20} />}
                  نمایش بیشتر
                </button>
              </div>
            )}
          </>
        )}
      </div>
    </section>
  );
};

export default Portfolio;