import React, { useState, useEffect } from 'react';
import { MessageSquare, PenTool, Code2, Rocket, Layers, CheckCircle } from 'lucide-react';
import { DataService } from '../services/dataService';
import { ProcessStep } from '../types';

const iconMap: Record<string, React.ReactNode> = {
  'MessageSquare': <MessageSquare size={24} className="md:w-8 md:h-8" />,
  'PenTool': <PenTool size={24} className="md:w-8 md:h-8" />,
  'Code2': <Code2 size={24} className="md:w-8 md:h-8" />,
  'Rocket': <Rocket size={24} className="md:w-8 md:h-8" />,
  'Layers': <Layers size={24} className="md:w-8 md:h-8" />,
  'CheckCircle': <CheckCircle size={24} className="md:w-8 md:h-8" />,
};

const Process: React.FC = () => {
  const [steps, setSteps] = useState<ProcessStep[]>([]);

  useEffect(() => {
    const update = async () => {
      const data = await DataService.getProcessSteps();
      setSteps(data);
    };
    update();
    window.addEventListener('data-updated', update);
    return () => window.removeEventListener('data-updated', update);
  }, []);

  return (
    <section className="py-16 md:py-24 bg-white dark:bg-slate-900 relative overflow-hidden transition-colors duration-300">
      {/* Connecting Line (Desktop) */}
      <div className="hidden lg:block absolute top-1/2 left-0 w-full h-0.5 bg-gradient-to-r from-transparent via-indigo-200 dark:via-indigo-800 to-transparent -translate-y-12 z-0"></div>

      <div className="container mx-auto px-4 relative z-10">
        <div className="text-center mb-12 md:mb-20">
          <span className="text-indigo-600 dark:text-indigo-400 font-bold tracking-wider text-sm uppercase">Workflow</span>
          <h2 className="text-2xl md:text-4xl font-bold text-slate-900 dark:text-white mt-2">مراحل انجام پروژه</h2>
        </div>

        {/* Desktop Grid */}
        <div className="hidden md:grid md:grid-cols-2 lg:grid-cols-4 gap-8">
          {steps.map((step) => (
            <div key={step.id} className="relative group">
              <div className="flex flex-col items-center text-center">
                <div className="w-20 h-20 rounded-2xl bg-white dark:bg-slate-800 border border-slate-200 dark:border-slate-700 flex items-center justify-center text-indigo-500 dark:text-indigo-400 mb-6 group-hover:bg-indigo-600 group-hover:text-white group-hover:scale-110 transition-all duration-300 shadow-xl shadow-indigo-500/10 dark:shadow-indigo-900/20 z-10 relative">
                   {iconMap[step.iconName] || iconMap['Layers']}
                   <div className="absolute inset-0 rounded-2xl bg-indigo-500/20 animate-ping opacity-0 group-hover:opacity-100 transition-opacity"></div>
                </div>
                <h3 className="text-xl font-bold text-slate-900 dark:text-white mb-3">{step.title}</h3>
                <p className="text-slate-600 dark:text-slate-400 text-sm leading-relaxed px-2">
                  {step.description}
                </p>
              </div>
            </div>
          ))}
        </div>

        {/* Mobile Vertical Timeline (App-like Tracking) */}
        <div className="md:hidden space-y-0 relative">
          {/* Vertical Line */}
          <div className="absolute right-6 top-4 bottom-4 w-0.5 bg-indigo-100 dark:bg-slate-800"></div>

          {steps.map((step, index) => (
            <div key={step.id} className="relative flex gap-6 pb-8 last:pb-0">
              {/* Icon Bubble */}
              <div className="relative z-10 shrink-0">
                <div className="w-12 h-12 rounded-full bg-white dark:bg-slate-800 border-4 border-slate-50 dark:border-slate-900 flex items-center justify-center text-indigo-600 dark:text-indigo-400 shadow-lg ring-1 ring-indigo-100 dark:ring-indigo-900/30">
                   {iconMap[step.iconName] || iconMap['Layers']}
                </div>
              </div>

              {/* Text Card */}
              <div className="flex-1 pt-1">
                <div className="bg-slate-50 dark:bg-slate-800/40 border border-slate-100 dark:border-slate-800 p-4 rounded-2xl rounded-tr-sm">
                   <h3 className="text-lg font-bold text-slate-900 dark:text-white mb-1 flex items-center gap-2">
                     <span className="text-indigo-500 text-xs">۰{index + 1}.</span>
                     {step.title}
                   </h3>
                   <p className="text-slate-600 dark:text-slate-400 text-xs leading-relaxed">
                     {step.description}
                   </p>
                </div>
              </div>
            </div>
          ))}
        </div>

      </div>
    </section>
  );
};

export default Process;