import React, { useState, useEffect } from 'react';
import { Star, Quote } from 'lucide-react';
import { DataService } from '../services/dataService';
import { Review } from '../types';

const Reviews: React.FC = () => {
  const [reviews, setReviews] = useState<Review[]>([]);

  useEffect(() => {
    const updateReviews = async () => {
      const data = await DataService.getReviews();
      setReviews(data);
    };
    updateReviews();
    window.addEventListener('data-updated', updateReviews);
    return () => window.removeEventListener('data-updated', updateReviews);
  }, []);

  return (
    <section className="py-16 md:py-24 bg-slate-50 dark:bg-slate-900 transition-colors duration-300">
      <div className="container mx-auto md:px-4">
        <div className="text-center mb-10 md:mb-16 px-4">
          <h2 className="text-2xl md:text-5xl font-bold text-slate-900 dark:text-white mb-4">نظرات مشتریان</h2>
          <div className="h-1 w-24 bg-indigo-600 mx-auto rounded-full"></div>
        </div>

        {/* Mobile: Horizontal Scroll | Desktop: Grid */}
        <div className="flex overflow-x-auto gap-4 px-4 pb-8 snap-x snap-mandatory hide-scrollbar md:grid md:grid-cols-3 md:gap-8 md:pb-0">
          {reviews.map((review) => (
            <div 
              key={review.id} 
              className="min-w-[300px] w-[85vw] md:w-auto snap-center flex-shrink-0 bg-white dark:bg-slate-800/50 p-6 md:p-8 rounded-3xl border border-slate-200 dark:border-slate-700 relative hover:bg-slate-50 dark:hover:bg-slate-800 transition-colors duration-300 shadow-sm hover:shadow-xl"
            >
              <Quote className="absolute top-6 left-6 text-slate-200 dark:text-slate-700 w-8 h-8" />
              <div className="flex gap-1 mb-4 md:mb-6">
                {[...Array(5)].map((_, i) => (
                  <Star key={i} size={16} className="text-yellow-400 fill-yellow-400" />
                ))}
              </div>
              <p className="text-slate-600 dark:text-slate-300 leading-relaxed mb-6 min-h-[60px] md:min-h-[80px] text-sm md:text-base">
                "{review.text}"
              </p>
              <div className="flex items-center gap-4 pt-4 border-t border-slate-100 dark:border-slate-700/50">
                <img src={review.avatar} alt={review.name} className="w-10 h-10 md:w-12 md:h-12 rounded-full object-cover border-2 border-indigo-500/50" />
                <div>
                  <h4 className="text-slate-900 dark:text-white font-bold text-sm">{review.name}</h4>
                  <p className="text-slate-500 text-xs mt-0.5">{review.role}</p>
                </div>
              </div>
            </div>
          ))}
           {/* Spacer */}
          <div className="w-1 flex-shrink-0 md:hidden"></div>
        </div>
      </div>
    </section>
  );
};

export default Reviews;