import React, { useRef } from 'react';
import { Bold, Italic, List, Heading2, Link, Code, Quote, Image } from 'lucide-react';

interface RichTextEditorProps {
  value: string;
  onChange: (value: string) => void;
  placeholder?: string;
}

const RichTextEditor: React.FC<RichTextEditorProps> = ({ value, onChange, placeholder }) => {
  const textareaRef = useRef<HTMLTextAreaElement>(null);

  const insertTag = (openTag: string, closeTag: string) => {
    const textarea = textareaRef.current;
    if (!textarea) return;

    const start = textarea.selectionStart;
    const end = textarea.selectionEnd;
    const text = textarea.value;
    const selectedText = text.substring(start, end);

    const before = text.substring(0, start);
    const after = text.substring(end);

    const newValue = `${before}${openTag}${selectedText}${closeTag}${after}`;
    
    onChange(newValue);

    // Restore focus and selection
    setTimeout(() => {
      textarea.focus();
      textarea.setSelectionRange(start + openTag.length, end + openTag.length);
    }, 0);
  };

  return (
    <div className="border border-slate-200 dark:border-slate-700 rounded-xl overflow-hidden bg-slate-50 dark:bg-slate-900 focus-within:ring-2 focus-within:ring-indigo-500 transition-all">
      {/* Toolbar */}
      <div className="flex flex-wrap gap-1 p-2 border-b border-slate-200 dark:border-slate-700 bg-white dark:bg-slate-800">
        <button type="button" onClick={() => insertTag('<strong>', '</strong>')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="Bold">
          <Bold size={18} />
        </button>
        <button type="button" onClick={() => insertTag('<em>', '</em>')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="Italic">
          <Italic size={18} />
        </button>
        <div className="w-px h-6 bg-slate-200 dark:bg-slate-700 mx-1 self-center"></div>
        <button type="button" onClick={() => insertTag('<h2>', '</h2>')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="Heading 2">
          <Heading2 size={18} />
        </button>
        <button type="button" onClick={() => insertTag('<ul>\n  <li>', '</li>\n</ul>')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="List">
          <List size={18} />
        </button>
        <button type="button" onClick={() => insertTag('<blockquote>', '</blockquote>')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="Quote">
          <Quote size={18} />
        </button>
        <div className="w-px h-6 bg-slate-200 dark:bg-slate-700 mx-1 self-center"></div>
        <button type="button" onClick={() => insertTag('<a href="#">', '</a>')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="Link">
          <Link size={18} />
        </button>
        <button type="button" onClick={() => insertTag('<img src="', '" alt="" />')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="Image">
          <Image size={18} />
        </button>
         <button type="button" onClick={() => insertTag('<pre><code>', '</code></pre>')} className="p-2 hover:bg-slate-100 dark:hover:bg-slate-700 rounded text-slate-600 dark:text-slate-300" title="Code">
          <Code size={18} />
        </button>
      </div>
      
      <textarea
        ref={textareaRef}
        value={value}
        onChange={(e) => onChange(e.target.value)}
        rows={12}
        className="w-full p-4 bg-transparent outline-none text-slate-900 dark:text-white font-mono text-sm resize-y"
        placeholder={placeholder}
      />
    </div>
  );
};

export default RichTextEditor;